import datetime
from math import sin, cos, pi, atan2, atan, asin, tan
import matplotlib
import matplotlib.pyplot as plt


# written by M. Wack 2009, 2010

# tested with Python 3,4

# sun azimuth is important for orienting samples with the so called 'sun compass'

# calc position of the sun in the sky (floating point)
def sunpso(lat, lon, time):
    '''calculate declination of the sun for a given geographic postion and time
    based on the procedure of the 1996 Astronomical Almanac (accurate to 0.01 deg 1950-2050).
    See also http://magician.ucsd.edu/Essentials/WebBookse112.html
    http://www.stargazing.net/kepler/sun.html'''

    # calculate days since 1.1.2000 12:00:00 as float
    td = time - datetime.datetime( 2000, 1, 1, 12, 0, 0)
    d = td.days + td.seconds / 3600.0 / 24

    # calc the mean longitude lm of the sun in range 0-360 deg
    j, lm = divmod( 280.460 + 0.9856474 * d, 360)

    # calc the mean anomaly g of the sun in range 0-360deg
    j, g = divmod( 357.528 + 0.9856003 * d, 360)

    # calc the ecliptic longitude le of the sun
    le = lm + 1.915 * sin( g / 180 * pi) + 0.020 * sin( 2 * g / 180 * pi)

    # calc obliquity of the ecliptic plane e
    e = 23.439 - 0.0000004 * d

    # find right ascension ra of the sun
    ra = atan2( cos( e / 180 * pi) * sin( le / 180 * pi), cos( le / 180 * pi)) * 180 / pi

    # find declination de of the sun
    de = asin( sin( e / 180 *pi) * sin( le / 180 * pi)) * 180 / pi

    # find local sidereal time in range 0-360
    j, lst = divmod( 280.46061837 + 360.98564736629 * d + lon, 360)

    #find hour angle of the sun in range 0-360
    j, ha = divmod( lst - ra, 360)

    # calc altitude alt of the sun
    alt = asin( sin( de / 180 * pi) * sin( lat / 180 * pi) + cos( lat / 180 * pi) * cos( ha / 180 * pi) * cos( de / 180 * pi)) * 180 / pi

    # calc azimuth az of the sun
    n = cos( ha / 180 * pi) * sin( lat / 180 * pi) - tan( de / 180 * pi) * cos( lat / 180 * pi)
    az = atan( sin( ha / 180 * pi) / n ) * 180 / pi
    if n < 0: az += 180
    az -= 180
    if az < 0: az += 360

    # return position of the sun
    return az, alt


if __name__ == "__main__":
    # set start time

    # list of time steps
    #startdate = datetime.datetime(2015, 12, 1, 0, 0, 0)
    t = datetime.date.today()
    startdate = datetime.datetime( t.year, t.month, t.day)
    time_list = [startdate + datetime.timedelta(minutes=t*10) for t in range(6*24)]

    az_list, alt_list = [], []

    lat = 48.15
    lon = 11.57

    # calculate sun position for all time steps
    for t in time_list:
        az, alt = sunpso(lat, lon, t)
        az_list.append(az)
        alt_list.append(alt)

    # plot the results
    ax1 = plt.subplot(211)
    ax1.set_title("Sun Position at {} / {} on {}".format(lat, lon, startdate.strftime("%d.%m.%Y")))
    ax1.plot_date(matplotlib.dates.date2num(time_list), az_list, ' .r', label="solar azimuth")
    ax1.xaxis.set_major_formatter(matplotlib.dates.DateFormatter('%H:%M'))
    ax1.set_xlabel("UTC")
    ax1.set_ylabel("solar azimuth")

    ax12 = ax1.twinx()
    ax12.plot_date(matplotlib.dates.date2num(time_list), alt_list, ' .b', label="solar altitude")
    ax12.set_ylabel("solar altitude")



    ax2 = plt.subplot(212)
    # plot azimuth vs altitude
    ax2.plot(az_list, alt_list, ' .b')
    ax2.set_xlim([0,360])
    ax2.set_xlabel("solar azimuth")
    ax2.set_ylabel("solar altitude")


    plt.show()