#!/bin/bash

#---------
# GMT script to calculate the continuous 2d strain rate tensor from 2d velocity files
#---------

# GMT wrapper script?
gmt=$(which GMT)

# set output names
name=california
velo=${name}_velocity.ps
velo_field_east=${name}_velo_field_east.ps
velo_field_north=${name}_velo_field_north.ps
strain_tensor=${name}_strain_rate_tensor.ps
strain_rate_crosses=${name}_strain_rate_crosses.ps
shear_strain_rate=${name}_shear_strain_rate.ps
dilatation_strain_rate=${name}_dilatation_strain_rate.ps
rotation_strain_rate=${name}_rotation_strain_rate.ps

# Maps to be plotted
velo_map=1
velo_field_maps=1
tensor_maps=1
strain_crosses_map=1
shear_strain_map=1
dilatation_strain_map=1
rotation_map=1

$gmt gmtset PAPER_MEDIA = A3

# Coordinates of maps
NORTH=42
SOUTH=31
WEST=-126
EAST=-113

# Definition of some variables
# path of velofile in psvelo format
velofile=velo_all_pos
# number of header lines in velofile
header=0

# path of fault file $gmt psxy format (> lon lat)
faultfile1=DATA/1Cal_*
# path of fault file $gmt psxy format (> lon lat)
faultfile2=DATA/Cal_*
# path of fault file $gmt psxy format (> lon lat)
faultfile3=DATA/21Cal_*
# path of fault file $gmt psxy format (> lon lat)
faultfile4=DATA/CA-BajaN.asc

# path of fault file $gmt psxy format (> lon lat)
plateboundary=DATA/platebound.gmt
# grid size in DEG
gridsize=0.04
# tension factor for the velocity component interpolation
tension=0.3
# determines in how many cells strain crosses are going to be plotted; e.g. every 15th cell
cross='1~17p'
# scaling factor for direction of max shear strain
crosssize=0.008
# scaling factor for colorbar of strain rate magnitude
orderofmagnitude=1000

arrow=`echo "-Se0.03/0.95 -A0.03/0.12/0.09"`

# 1 to plot topography
topo=1
# path of DEM
topofile=DATA/NW.grd
# resolution of DEM in arcsec
topores=60

function calc () {
    awk "BEGIN { print $* ; }"
}

mask=`calc "$gridsize * 15"`
magnitude=`calc " 1/(1000*${orderofmagnitude})" | awk '{printf("%1.1e",$0)}'`


#---------
# read in velofile and interpolate velocity
#---------
awk '(NR>'${header}'){print($1, $2, $3, $5)}' ${velofile} | $gmt blockmedian -I$gridsize -R$WEST/$EAST/$SOUTH/$NORTH -V > east_velo_median
$gmt surface east_velo_median -Geast_velo.grd -I$gridsize -R$WEST/$EAST/$SOUTH/$NORTH -T$tension -V

awk '(NR>'${header}'){print($1, $2, $4, $6)}' ${velofile} | $gmt blockmedian -I$gridsize -R$WEST/$EAST/$SOUTH/$NORTH -V > north_velo_median
$gmt surface north_velo_median -Gnorth_velo.grd -I$gridsize -R$WEST/$EAST/$SOUTH/$NORTH -T$tension -V


#---------
# calculate velo gradient
#---------
$gmt grdgradient east_velo.grd -Gtmp.grd -A270 -V -M
$gmt grdmath -V tmp.grd $orderofmagnitude MUL = e_e.grd
$gmt grdgradient east_velo.grd -Gtmp.grd -A180 -V -M
$gmt grdmath -V tmp.grd $orderofmagnitude MUL = e_n.grd
$gmt grdgradient north_velo.grd -Gtmp.grd -A270 -V -M
$gmt grdmath -V tmp.grd $orderofmagnitude MUL = n_e.grd
$gmt grdgradient north_velo.grd -Gtmp.grd -A180 -V -M
$gmt grdmath -V tmp.grd $orderofmagnitude MUL = n_n.grd

# i,j component of strain tensor (mean of e_n and n_e component):
$gmt grdmath -V e_n.grd n_e.grd ADD 0.5 MUL = mean_e_n.grd


#---------
# calc eigenvalues, max shear strain rate, and dilatational strain rate
#---------
$gmt grdmath -V e_e.grd n_n.grd ADD e_e.grd n_n.grd SUB 2 POW mean_e_n.grd 2 POW 4 MUL ADD SQRT ADD 2 DIV = lambda1.grd
$gmt grdmath -V e_e.grd n_n.grd ADD e_e.grd n_n.grd SUB 2 POW mean_e_n.grd 2 POW 4 MUL ADD SQRT SUB 2 DIV = lambda2.grd
$gmt grd2xyz lambda1.grd > lambda1.xyz
$gmt grd2xyz lambda2.grd > lambda2.xyz
$gmt grdmath -V lambda1.grd lambda2.grd SUB 2 DIV = max_shear.grd

$gmt grdmath -V lambda1.grd lambda2.grd ADD = str_dilatational.grd

#---------
# calc direction of max_shear
#---------
$gmt grd2xyz max_shear.grd > max_shear.xyz
$gmt grdmath -V 90 0.5 2 mean_e_n.grd MUL e_e.grd n_n.grd SUB DIV 1 ATAN2 MUL 180 MUL 3.14 DIV SUB 45 ADD = phi1.grd
$gmt grd2xyz phi1.grd > phi1.xyz

$gmt grdmath -V 90 lambda2.grd e_e.grd SUB mean_e_n.grd DIV 1 ATAN2 180 MUL 3.14 DIV SUB = phi2.grd
$gmt grd2xyz phi2.grd > phi2.xyz
paste lambda1.xyz lambda2.xyz phi2.xyz | awk '{print($1, $2, $3, $6, $9)}' | sed -n $cross > phi_shear.xyl1l2p

paste max_shear.xyz phi1.xyz | awk '{print($1, $2, $3, $6)}' | sed -n $cross > phi_shear.xysp
	awk '
		function acos(x) { return atan2((1.-x^2)^0.5,x) }
		function asin(x) { return atan2(x,(1.-x^2)^0.5) }
		{
		pi = atan2(0,-1); lat = $2; lon = $1; alpha = $4*pi/180; a = $3*'${crosssize}';
		lat_right = 90 - acos(cos(a)*cos((90 - lat)*pi/180) + sin(a)*sin((90 - lat)*pi/180)*cos(alpha)) *180/pi;
		lon_right = lon + asin(sin(a)/sin((90-lat_right)*pi/180) * sin(alpha)) * 180/pi;
		lat_left = 90 - acos(cos(a)*cos((90 - lat)*pi/180) + sin(a)*sin((90 - lat)*pi/180)*cos(alpha-pi)) *180/pi;
		lon_left = lon - asin(sin(a)/sin((90-lat_right)*pi/180) * sin(alpha)) * 180/pi;}
		{printf ("> -Z%.2f\n %9.5f %9.5f \n %9.5f %9.5f \n %9.5f %9.5f \n", a, lon_left, lat_left, lon, lat, lon_right, lat_right)}' phi_shear.xysp > dir1
	awk '
		function acos(x) { return atan2((1.-x^2)^0.5,x) }
		function asin(x) { return atan2(x,(1.-x^2)^0.5) }
		{
		pi = atan2(0,-1); lat = $2; lon = $1; alpha = $4*pi/180+pi/2; a = $3*'${crosssize}'; 
		lat_right = 90 - acos(cos(a)*cos((90 - lat)*pi/180) + sin(a)*sin((90 - lat)*pi/180)*cos(alpha)) *180/pi;
		lon_right = lon + asin(sin(a)/sin((90-lat_right)*pi/180) * sin(alpha)) * 180/pi;
		lat_left = 90 - acos(cos(a)*cos((90 - lat)*pi/180) + sin(a)*sin((90 - lat)*pi/180)*cos(alpha-pi)) *180/pi;
		lon_left = lon - asin(sin(a)/sin((90-lat_right)*pi/180) * sin(alpha)) * 180/pi;}
		{printf ("> -Z%.2f\n %9.5f %9.5f \n %9.5f %9.5f \n %9.5f %9.5f \n", a, lon_left, lat_left, lon, lat, lon_right, lat_right)}' phi_shear.xysp > dir2

#---------
# calc rotational strain rate
#---------
$gmt grdmath -V n_e.grd e_n.grd SUB 0.5 MUL = omega.grd


#---------
# prepare dem
#---------
if [ "${topo}" = "1" ]
then
	$gmt grdcut -R ${topofile} -Gtopo.grd
	$gmt grdgradient topo.grd -A135 -Ne0.8 -Gshadow.grd
fi


#---------
# plot velocity file
#---------
if [ "${velo_map}" = "1" ]
then
	$gmt psbasemap -R$WEST/$EAST/$SOUTH/$NORTH -JM25c -B2g1 -V -K > $velo
	
	if [ "${topo}" = "1" ]
	then
		$gmt grdimage topo.grd -R -CGMT_topo.cpt -J -O -K -V >> $velo
		$gmt pscoast -R -J -B2g1 -W1.0p -Dh -Ir -O -V -N1/1p -N2/0.5p -K >> $velo
	else
		$gmt pscoast -R -J -G200/200/200 -S200/255/255 -B2g1 -W1.0p -Dh -Ir -O -V -N1/1p -N2/0.5p -K >> $velo
	fi

	$gmt psxy $faultfile1 -R -J -M -Wthinnest/brown -H -O -K >> $velo
	$gmt psxy $faultfile2 -R -J -M -Wthinnest/brown -H -O -K >> $velo
	$gmt psxy $plateboundary -R -J -M -Wthicker/brown -H -O -K >> $velo
	
	#awk '($2<"33"){print($0)}' $velofile | $gmt psvelo -R -J -W0.02p,blue -Gblue -Se0.05/0.95 -A0.03/0.12/0.09 -H${header} -O -V -K >> $velo

	$gmt psvelo $velofile -R -J -W0.02p,blue -Gblue $arrow -H${header} -O -V -K >> $velo
	lon=`calc " $WEST + 0.5 "`
	lat=`calc " $SOUTH + 0.5 "`
	echo "$lon $lat -20 0 0 0 0" | $gmt psvelo -R -J -W0.02p $arrow -O -V -K >> $velo
	echo "$lon $lat 18 0 5 LM 20 mm/yr" | $gmt pstext -R -J -O -V >> $velo
fi

#---------
# plot velocity fields
#---------
if [ "${velo_field_maps}" = "1" ]
then
	$gmt psbasemap -R$WEST/$EAST/$SOUTH/$NORTH -JM25c -B2g1 -V -K > $velo_field_east
	$gmt psbasemap -R$WEST/$EAST/$SOUTH/$NORTH -JM25c -B2g1 -V -K > $velo_field_north

	if [ "${topo}" = "1" ]
	then
		$gmt grdsample east_velo.grd -I${topores}c -Geast_velo_sampled.grd -V																		# put -T if it does not work
		$gmt grdsample north_velo.grd -I${topores}c -Gnorth_velo_sampled.grd -V
		$gmt grdimage -R topo.grd -Ishadow.grd -Crelief.cpt -J -O -K -V >> $velo_field_east
		$gmt grdimage -R topo.grd -Ishadow.grd -Crelief.cpt -J -O -K -V >> $velo_field_north
		awk '(NR>'${header}'){print($1, $2)}' $velofile | $gmt psmask -R -I$gridsize -J -S$mask -O -K -Gwhite >> $velo_field_east
		awk '(NR>'${header}'){print($1, $2)}' $velofile | $gmt psmask -R -I$gridsize -J -S$mask -O -K -Gwhite >> $velo_field_north
		$gmt grdimage -R east_velo_sampled.grd -Ishadow.grd -CGMT_jet_velo.cpt -J -O -K -V >> $velo_field_east
		$gmt grdimage -R north_velo_sampled.grd -Ishadow.grd -CGMT_jet_velo.cpt -J -O -K -V >> $velo_field_north
	else
		$gmt pscoast -R -J -B2g1 -W1.0p -G200/200/200 -Dh -Ir -O -V -N1/1p -N2/0.5p -K >> $velo_field_east
		$gmt pscoast -R -J -B2g1 -W1.0p -G200/200/200 -Dh -Ir -O -V -N1/1p -N2/0.5p -K >> $velo_field_north
		awk '(NR>'${header}'){print($1, $2)}' $velofile | $gmt psmask -R -I$gridsize -J -S$mask -O -K -Gwhite >> $velo_field_east
		awk '(NR>'${header}'){print($1, $2)}' $velofile | $gmt psmask -R -I$gridsize -J -S$mask -O -K -Gwhite >> $velo_field_north
		$gmt grdimage -R east_velo.grd -CGMT_jet_velo.cpt -J -O -K -V >> $velo_field_east
		$gmt grdimage -R north_velo.grd -CGMT_jet_velo.cpt -J -O -K -V >> $velo_field_north
	fi
	$gmt psmask -C -O -K >> $velo_field_east
	$gmt psmask -C -O -K >> $velo_field_north

	$gmt pscoast -R -J -B2g1 -S200/255/255 -W1.0p -Dh -Ir -O -V -N1/1p -N2/0.5p -K >> $velo_field_east
	$gmt pscoast -R -J -B2g1 -S200/255/255 -W1.0p -Dh -Ir -O -V -N1/1p -N2/0.5p -K >> $velo_field_north

	$gmt psxy $faultfile1 -R -J -M -Wthinnest/brown -H -O -K >> $velo_field_east
	$gmt psxy $faultfile2 -R -J -M -Wthinnest/brown -H -O -K >> $velo_field_east
	$gmt psxy $plateboundary -R -J -M -Wthicker/brown -H -O -K >> $velo_field_east
	$gmt psxy $faultfile1 -R -J -M -Wthinnest/brown -H -O -K >> $velo_field_north
	$gmt psxy $faultfile2 -R -J -M -Wthinnest/brown -H -O -K >> $velo_field_north
	$gmt psxy $plateboundary -R -J -M -Wthicker/brown -H -O -K >> $velo_field_north

	$gmt psscale -D27c/10c/7c/1c -CGMT_jet_velo.cpt -L -O -K -V >> $velo_field_east
	$gmt psscale -D27c/10c/7c/1c -CGMT_jet_velo.cpt -L -O -K -V >> $velo_field_north

	lon=`calc " $EAST + 1 "`
	lat=`calc " $SOUTH + 2 "`
	echo "$lon $lat 18 0 5 LM mm/yr" | $gmt pstext -R -J -N -O -V >> $velo_field_east
	echo "$lon $lat 18 0 5 LM mm/yr" | $gmt pstext -R -J -N -O -V >> $velo_field_north
fi

#---------
# plot strain rate tensor
#---------
if [ "${tensor_maps}" = "1" ]
then
	lon=`calc " $WEST + 1 "`
	lat=`calc " $SOUTH + 2 "`

	$gmt pscoast -R$WEST/$EAST/$SOUTH/$NORTH -JM8c -B2NEws -W1.0p -G200/200/200 -Dh -Ir -V -N1/1p -N2/0.5p -K -X15c> $strain_tensor
	awk '(NR>'${header}'){print($1, $2)}' $velofile | $gmt psmask -R -I$gridsize -J -S$mask -O -K -Gwhite >> $strain_tensor
	$gmt grdimage -R n_n.grd -CGMT_jet.cpt -J -O -K -V >> $strain_tensor
	$gmt psmask -C -O -K >> $strain_tensor
	$gmt pscoast -R -J -B2NEwsg1 -W1.0p -Dh -Ir -O -V -N1/1p -N2/0.5p -K >> $strain_tensor
	$gmt psxy $faultfile1 -R -J -M -Wthinnest/brown -H -O -K >> $strain_tensor
	$gmt psxy $faultfile2 -R -J -M -Wthinnest/brown -H -O -K >> $strain_tensor
	$gmt psxy $plateboundary -R -J -M -Wthin/brown -H -O -K >> $strain_tensor
#	echo "$lon $lat 20 0 5 LM  @~e@~@-ee" | $gmt pstext -R -J -N -O -V -K >> $strain_tensor

	$gmt pscoast -R$WEST/$EAST/$SOUTH/$NORTH -JM8c -B2NEws -W1.0p -G200/200/200 -Dh -Ir -O -V -N1/1p -N2/0.5p -K -Y10c >> $strain_tensor
	awk '(NR>'${header}'){print($1, $2)}' $velofile | $gmt psmask -R -I$gridsize -J -S$mask -O -K -Gwhite >> $strain_tensor
	$gmt grdimage -R mean_e_n.grd -CGMT_jet.cpt -J -O -K -V >> $strain_tensor
	$gmt psmask -C -O -K >> $strain_tensor
	$gmt pscoast -R -J -B2NEwsg1 -W1.0p -Dh -Ir -O -V -N1/1p -N2/0.5p -K >> $strain_tensor
	$gmt psxy $faultfile1 -R -J -M -Wthinnest/brown -H -O -K >> $strain_tensor
	$gmt psxy $faultfile2 -R -J -M -Wthinnest/brown -H -O -K >> $strain_tensor
	$gmt psxy $plateboundary -R -J -M -Wthin/brown -H -O -K >> $strain_tensor

	$gmt pscoast -R$WEST/$EAST/$SOUTH/$NORTH -JM8c -B2NEws -W1.0p -G200/200/200 -Dh -Ir -O -V -N1/1p -N2/0.5p -K -X-9.5c >> $strain_tensor
	awk '(NR>'${header}'){print($1, $2)}' $velofile | $gmt psmask -R -I$gridsize -J -S$mask -O -K -Gwhite >> $strain_tensor
	$gmt grdimage -R e_e.grd -CGMT_jet.cpt -J -O -K -V >> $strain_tensor
	$gmt psmask -C -O -K >> $strain_tensor
	$gmt pscoast -R -J -B2NEwsg1 -W1.0p -Dh -Ir -O -V -N1/1p -N2/0.5p -K >> $strain_tensor
	$gmt psxy $faultfile1 -R -J -M -Wthinnest/brown -H -O -K >> $strain_tensor
	$gmt psxy $faultfile2 -R -J -M -Wthinnest/brown -H -O -K >> $strain_tensor
	$gmt psxy $plateboundary -R -J -M -Wthin/brown -H -O -K >> $strain_tensor

	$gmt psscale -D2c/3.5c/6c/1c -CGMT_jet.cpt -L -O -K -V -Y-8c >> $strain_tensor

	echo "$lon $lat 18 0 5 LM @~m@~strain/yr" | $gmt pstext -R -J -N -O -V -Y1.7c -X3.8c >> $strain_tensor
fi

#---------
# plot strain crosses
#---------
if [ "${strain_crosses_map}" = "1" ]
then
	$gmt psbasemap -R$WEST/$EAST/$SOUTH/$NORTH -JM25c -B2g1 -V -K > $strain_rate_crosses
	
	if [ "${topo}" = "1" ]
	then
		$gmt grdimage topo.grd -R -CGMT_topo.cpt -J -O -K -V >> $strain_rate_crosses
		awk '(NR>'${header}'){print($1, $2)}' $velofile | $gmt psmask -R -I$gridsize -J -S$mask -O -K -Gwhite >> $strain_rate_crosses
			$gmt grdimage ${topofile} -R -CGMT_topo.cpt -J -O -K -V >> $strain_rate_crosses
			$gmt psvelo phi_shear.xyl1l2p -R -J -Sx0.5c -Wblue -O -V -K >> $strain_rate_crosses
		$gmt psmask -C -O -K >> $strain_rate_crosses
		$gmt pscoast -R -J -S200/255/255 -B2g1 -W1.0p -Dh -Ir -O -V -N1/1p -N2/0.5p -K >> $strain_rate_crosses
	else
		$gmt pscoast -R -J -B2g1 -G200/200/200 -W1.0p -Dh -Ir -O -V -N1/1p -N2/0.5p -K >> $strain_rate_crosses
		awk '(NR>'${header}'){print($1, $2)}' $velofile | $gmt psmask -R -I$gridsize -J -S$mask -O -K -Gwhite >> $strain_rate_crosses
			$gmt pscoast -R -J -B2g1 -G200/200/200 -W1.0p -Dh -Ir -O -V -N1/1p -N2/0.5p -K >> $strain_rate_crosses
			$gmt psvelo phi_shear.xyl1l2p -R -J -Sx0.5c -Wblue -O -V -K >> $strain_rate_crosses
		$gmt psmask -C -O -K >> $strain_rate_crosses
		$gmt pscoast -R -J -S200/255/255 -B2g1 -W1.0p -Dh -Ir -O -V -N1/1p -N2/0.5p -K >> $strain_rate_crosses
	fi
	
	$gmt psxy $faultfile1 -R -J -M -Wthinnest/brown -H -O -K >> $strain_rate_crosses
	$gmt psxy $faultfile2 -R -J -M -Wthinnest/brown -H -O -K >> $strain_rate_crosses
	$gmt psxy $plateboundary -R -J -M -Wthick/brown -H -O -K >> $strain_rate_crosses
	# plot site position
	awk '(NR>'${header}'){print($1, $2)}' $velofile | $gmt psxy -R -J -Sc0.1c -Wblack -O -V -K >> $strain_rate_crosses
	
	lon=`calc " $WEST + 0.5 "`
	lat=`calc " $SOUTH + 0.5 "`
	lont=`calc " $WEST + 0.8 "`
	echo "$lon $lat 1 -1 0" | $gmt psvelo -R -J -Sx0.5c -Wblack -O -V -K >> $strain_rate_crosses
	echo "$lont $lat 18 0 5 LM ${magnitude} strain/yr" | $gmt pstext -R -J -O -V >> $strain_rate_crosses
fi

#---------
# plot shear strain rate
#---------
if [ "${shear_strain_map}" = "1" ]
then
	$gmt psbasemap -R$WEST/$EAST/$SOUTH/$NORTH -JM25c -P -B2g1 -V -K > $shear_strain_rate

	if [ "${topo}" = "1" ]
	then
		# put -T if it does not work
		$gmt grdsample max_shear.grd -I${topores}c -Gmax_shear_sampled.grd -V
		$gmt grdimage -R topo.grd -Ishadow.grd -Crelief.cpt -J -O -K -V >> $shear_strain_rate
		awk '(NR>'${header}'){print($1, $2)}' $velofile | $gmt psmask -R -I$gridsize -J -S$mask -O -K -Gwhite >> $shear_strain_rate
			$gmt grdimage -R max_shear_sampled.grd -Ishadow.grd -CGMT_seis2.cpt -J -O -K -V >> $shear_strain_rate
	else
		$gmt pscoast -R -J -B2g1 -W1.0p -G200/200/200 -Dh -Ir -O -V -N1/1p -N2/0.5p -K >> $shear_strain_rate
		awk '(NR>'${header}'){print($1, $2)}' $velofile | $gmt psmask -R -I$gridsize -J -S$mask -O -K -Gwhite >> $shear_strain_rate
			$gmt grdimage -R max_shear.grd -CGMT_seis.cpt -J -O -K -V >> $shear_strain_rate
	fi
	$gmt psxy dir1 -R -J -M -W2 -O -K >> $shear_strain_rate
	$gmt psxy dir2 -R -J -M -W2 -O -K >> $shear_strain_rate
	$gmt psmask -C -O -K >> $shear_strain_rate
	
	$gmt pscoast -R -J -B2g1 -S200/255/255 -W1.0p -Dh -Ir -O -V -N1/1p -N2/0.5p -K >> $shear_strain_rate
	$gmt psxy $faultfile1 -R -J -M -Wthinnest/brown -H -O -K >> $shear_strain_rate
	$gmt psxy $faultfile2 -R -J -M -Wthinnest/brown -H -O -K >> $shear_strain_rate
	$gmt psxy $faultfile3 -R -J -M -Wthinnest/brown -H -O -K >> $shear_strain_rate
	$gmt psxy $faultfile4 -R -J -M -Wthinnest/brown -H -O -K >> $shear_strain_rate
	$gmt psxy $plateboundary -R -J -M -Wthin/brown -H -O -K >> $shear_strain_rate
	# plot site position
	awk '(NR>'${header}'){print($1, $2)}' $velofile | $gmt psxy -R -J -Sc0.1c -Wblack -O -V -K >> $shear_strain_rate
	
	$gmt psscale -D35.5c/15c/7c/1c -CGMT_seis.cpt -L -O -K -V >> $shear_strain_rate
	$gmt psxy boxes -R -J -Wthicker/black -L -M -O -V -K >> $shear_strain_rate

	lon1=`calc " $WEST + 0.2 "`
	lon2=`calc " $WEST + 3 "`
	lat1=`calc " $SOUTH + 0.2 "`
	lat2=`calc " $SOUTH + 0.8 "`
	$gmt psxy << EOF -R -J -G200/200/200 -Wthick/black -L -M -O -V -K >> $shear_strain_rate
	$lon1 $lat1
	$lon2 $lat1
	$lon2 $lat2
	$lon1 $lat2	
EOF
	
	lon=`calc " $WEST + 0.7 "`
	lat=`calc " $SOUTH + 0.5 "`
	lont=`calc " $WEST + 1 "`
	echo "$lon $lat 0.5 0" | awk '
			function acos(x) { return atan2((1.-x^2)^0.5,x) }
			function asin(x) { return atan2(x,(1.-x^2)^0.5) }
			{
			pi = atan2(0,-1); lat = $2; lon = $1; alpha = $4*pi/180; a = $3*'${crosssize}';
			lat_right = 90 - acos(cos(a)*cos((90 - lat)*pi/180) + sin(a)*sin((90 - lat)*pi/180)*cos(alpha)) *180/pi;
			lon_right = lon + asin(sin(a)/sin((90-lat_right)*pi/180) * sin(alpha)) * 180/pi;
			lat_left = 90 - acos(cos(a)*cos((90 - lat)*pi/180) + sin(a)*sin((90 - lat)*pi/180)*cos(alpha-pi)) *180/pi;
			lon_left = lon - asin(sin(a)/sin((90-lat_right)*pi/180) * sin(alpha)) * 180/pi;}
			{printf ("> -Z%.2f\n %9.5f %9.5f \n %9.5f %9.5f \n %9.5f %9.5f \n", a, lon_left, lat_left, lon, lat, lon_right, lat_right)}' | $gmt psxy -R -J -M -W3 -O -K >> $shear_strain_rate
	echo "$lon $lat 0.5 0" | awk '
			function acos(x) { return atan2((1.-x^2)^0.5,x) }
			function asin(x) { return atan2(x,(1.-x^2)^0.5) }
			{
			pi = atan2(0,-1); lat = $2; lon = $1; alpha = $4*pi/180+pi/2; a = $3*'${crosssize}';
			lat_right = 90 - acos(cos(a)*cos((90 - lat)*pi/180) + sin(a)*sin((90 - lat)*pi/180)*cos(alpha)) *180/pi;
			lon_right = lon + asin(sin(a)/sin((90-lat_right)*pi/180) * sin(alpha)) * 180/pi;
			lat_left = 90 - acos(cos(a)*cos((90 - lat)*pi/180) + sin(a)*sin((90 - lat)*pi/180)*cos(alpha-pi)) *180/pi;
			lon_left = lon - asin(sin(a)/sin((90-lat_right)*pi/180) * sin(alpha)) * 180/pi;}
			{printf ("> -Z%.2f\n %9.5f %9.5f \n %9.5f %9.5f \n %9.5f %9.5f \n", a, lon_left, lat_left, lon, lat, lon_right, lat_right)}' | $gmt psxy -R -J -M -W3 -O -K >> $shear_strain_rate
#	echo "$lont $lat 18 0 5 LM 0.5*${magnitude}/yr" | $gmt pstext -R -J -O -V -K >> $shear_strain_rate	
	echo "$lont $lat 18 0 5 LM 0.5 @~m@~strain/yr" | $gmt pstext -R -J -O -V -K >> $shear_strain_rate	
	psmeca ~/strainmap/earthquakes -R -J -O -V -G0 -C -Wthicker -Sc2.5 -K >> $shear_strain_rate
	lon=`calc " $EAST + 1 "`
	lat=`calc " $SOUTH + 3.5 "`
#	echo "$lon $lat 18 0 5 LM ${magnitude} strain/yr" | $gmt pstext -R -J -N -O -V >> $shear_strain_rate
	echo "$lon $lat 18 0 5 LM @~m@~strain/yr" | $gmt pstext -R -J -N -O -V >> $shear_strain_rate
fi
	
#---------
# plot dilatation strain rate
#---------
if [ "${dilatation_strain_map}" = "1" ]
then
	$gmt psbasemap -R$WEST/$EAST/$SOUTH/$NORTH -JM25c -B2g1 -V -K > $dilatation_strain_rate
	
	if [ "${topo}" = "1" ]
	then
		# put -T if it does not work
		$gmt grdsample str_dilatational.grd -I${topores}c -Gstr_dilatational_sampled.grd -V
		$gmt grdimage -R topo.grd -Ishadow.grd -Crelief.cpt -J -O -K -V >> $dilatation_strain_rate
		awk '(NR>'${header}'){print($1, $2)}' $velofile | $gmt psmask -R -I$gridsize -J -S$mask -O -K -Gwhite >> $dilatation_strain_rate
			$gmt grdimage -R str_dilatational_sampled.grd -Ishadow.grd -CGMT_jet2.cpt -J -O -K -V >> $dilatation_strain_rate
	else
		$gmt pscoast -R -J -B2g1 -W1.0p -G200/200/200 -Dh -Ir -O -V -N1/1p -N2/0.5p -K >> $dilatation_strain_rate
		awk '(NR>'${header}'){print($1, $2)}' $velofile | $gmt psmask -R -I$gridsize -J -S$mask -O -K -Gwhite >> $dilatation_strain_rate
			$gmt grdimage -R str_dilatational.grd -CGMT_jet2.cpt -J -O -K -V >> $dilatation_strain_rate
	fi
	$gmt psmask -C -O -K >> $dilatation_strain_rate
	
	$gmt pscoast -R -J -B2g1 -S200/255/255 -W1.0p -Dh -Ir -O -V -N1/1p -N2/0.5p -K >> $dilatation_strain_rate
	$gmt psxy $faultfile1 -R -J -M -Wthinnest/brown -H -O -K >> $dilatation_strain_rate
	$gmt psxy $faultfile2 -R -J -M -Wthinnest/brown -H -O -K >> $dilatation_strain_rate
	$gmt psxy $plateboundary -R -J -M -Wthick/brown -H -O -K >> $dilatation_strain_rate
	# plot site position
	awk '(NR>'${header}'){print($1, $2)}' $velofile | $gmt psxy -R -J -Sc0.1c -Wblack -O -V -K >> $dilatation_strain_rate
	
	$gmt psscale -D27c/10c/7c/1c -CGMT_jet2.cpt -L -O -K -V >> $dilatation_strain_rate
	
	lon=`calc " $EAST + 0.8 "`
	lat=`calc " $SOUTH + 2 "`
#	echo "$lon $lat 18 0 5 LM ${magnitude} strain/yr" | $gmt pstext -R -J -N -O -V >> $dilatation_strain_rate
	echo "$lon $lat 18 0 5 LM @~m@~strain/yr" | $gmt pstext -R -J -N -O -V >> $dilatation_strain_rate
fi
	
#---------
# plot rotational strain rate
#---------
if [ "${rotation_map}" = "1" ]
then
	$gmt psbasemap -R$WEST/$EAST/$SOUTH/$NORTH -JM25c -B2g1 -V -K > $rotation_strain_rate
	
	if [ "${topo}" = "1" ]
	then
		# put -T if it does not work
		$gmt grdsample omega.grd -I${topores}c -Gomega_sampled.grd -V
		$gmt grdimage -R topo.grd -Ishadow.grd -Crelief.cpt -J -O -K -V >> $rotation_strain_rate
		awk '(NR>'${header}'){print($1, $2)}' $velofile | $gmt psmask -R -I$gridsize -J -S$mask -O -K -Gwhite >> $rotation_strain_rate
			$gmt grdimage -R omega_sampled.grd -Ishadow.grd -CGMT_rot.cpt -J -O -K -V >> $rotation_strain_rate
	else
		$gmt pscoast -R -J -B2g1 -W1.0p -G200/200/200 -Dh -Ir -O -V -N1/1p -N2/0.5p -K >> $rotation_strain_rate
		awk '(NR>'${header}'){print($1, $2)}' $velofile | $gmt psmask -R -I$gridsize -J -S$mask -O -K -Gwhite >> $rotation_strain_rate
			$gmt grdimage -R omega.grd -CGMT_rot.cpt -J -O -K -V >> $rotation_strain_rate
	fi
	$gmt psmask -C -O -K >> $rotation_strain_rate
	
	$gmt pscoast -R -J -B2g1 -S200/255/255 -W1.0p -Dh -Ir -O -V -N1/1p -N2/0.5p -K >> $rotation_strain_rate
	$gmt psxy $faultfile1 -R -J -M -Wthinnest/brown -H -O -K >> $rotation_strain_rate
	$gmt psxy $faultfile2 -R -J -M -Wthinnest/brown -H -O -K >> $rotation_strain_rate
	$gmt psxy $plateboundary -R -J -M -Wthick/brown -H -O -K >> $rotation_strain_rate
	# plot site position
	awk '(NR>'${header}'){print($1, $2)}' $velofile | $gmt psxy -R -J -Sc0.1c -Wblack -O -V -K >> $rotation_strain_rate
	
	$gmt psscale -D27c/10c/7c/1c -CGMT_rot.cpt -L -O -K -V >> $rotation_strain_rate
	
	lon=`calc " $EAST + 0.8 "`
	lat=`calc " $SOUTH + 2 "`
#	echo "$lon $lat 18 0 5 LM rotation ${magnitude} strain/yr" | $gmt pstext -R -J -N -O -V >> $rotation_strain_rate
	echo "$lon $lat 18 0 5 LM @~m@~strain/yr" | $gmt pstext -R -J -N -O -V >> $rotation_strain_rate
fi	
#---------
# clean up
#---------
rm topo.grd shadow.grd max_shear_sampled.grd e_e.grd e_n.grd n_e.grd n_n.grd mean_e_n.grd lambda* max_shear.grd phi* max_shear.xyz dir1 dir2 east* north* tmp* str_* omega*

