!*****************************************************************************
!***************** Background Model Initialization ***************************
!*****************************************************************************

SUBROUTINE fd3s_model

    IMPLICIT NONE
    INCLUDE 'common_global.h'

    INTEGER xslab,yslab,lnblnk
    REAL vphomo(0:nx,0:ny,0:nz)  ! homogeneous P velocity   
    REAL vshomo(0:nx,0:ny,0:nz)  ! homogeneous shear velocity   
    REAL tonga(0:nx,0:ny,0:nz)   ! variable for slab initialization 

    WRITE(*,*)' Begin model '
    WRITE(99,*)' Begin model '

!=============================================================================

! homogeneous halfspace 

    IF(model_type==1)THEN

    WRITE(*,*)' Generating homogeneous model ...'
    WRITE(99,*)' Generating homogeneous model ...'

    rhoinv=1./rho0
    mu=rho0*vs0**2
    lam=rho0*vp0**2-2*mu


!=============================================================================

! PREM, spherically symmetric

    ELSEIF(model_type==2)THEN
        
    CALL prem

    ro=ro*1000. ! unit conversion to meter
    vs=vs*1000.
    vp=vp*1000.

    WRITE(99,*)MINVAL(ro),' < ro < ',MAXVAL(ro)
    WRITE(99,*)MINVAL(vp),' < vp < ',MAXVAL(vp)
    WRITE(99,*)MINVAL(vs),' < vs < ',MAXVAL(vs)

    DO iz=0,nz
      rhoinv(:,:,iz)=1/ro(iz)
      mu(:,:,iz)=ro(iz)*vs(iz)**2
      lam(:,:,iz)=ro(iz)*vp(iz)**2-2*mu(:,:,iz)
    ENDDO
 
         
!=============================================================================

! 2 homogeneous layers, discontinuity at 240km depth

     ELSEIF(model_type==3)THEN
         
     CALL homo_2layer

     WRITE(99,*)MINVAL(ro),' < ro < ',MAXVAL(ro)
     WRITE(99,*)MINVAL(vp),' < vp < ',MAXVAL(vp)
     WRITE(99,*)MINVAL(vs),' < vs < ',MAXVAL(vs)

     DO iz=0,nz
       rhoinv(:,:,iz)=1/ro(iz)
       mu(:,:,iz)=ro(iz)*vs(iz)**2
       lam(:,:,iz)=ro(iz)*vp(iz)**2-2*mu(:,:,iz)           
     ENDDO
              

!=============================================================================

! slab model 
! The model is constructed via matlab.
! Files are read in for each processor, respectively.

     ELSEIF (model_type==7) THEN

     WRITE(*,*)' Initialising TONGA slab model ....'
     WRITE(99,*)' Initialising TONGA slab model ....'

     CALL prem

     ro=ro*1000.
     vs=vs*1000.
     vp=vp*1000.

     WRITE(99,*)''
        
     WRITE(99,*)' PREM Min./Max. Values of Processor :'
 
     WRITE(99,*)MINVAL(ro),' < ro < ',MAXVAL(ro)
     WRITE(99,*)MINVAL(vp),' < vp < ',MAXVAL(vp)
     WRITE(99,*)MINVAL(vs),' < vs < ',MAXVAL(vs)
        
     DO iz=0,nz
       rhoinv(:,:,iz)=1/ro(iz)                     ! PREM elastic parameters
       mu(:,:,iz)=ro(iz)*vs(iz)**2
       lam(:,:,iz)=ro(iz)*vp(iz)**2-2*mu(:,:,iz)
     ENDDO
          
     vphomo=SQRT((lam+2*mu)*rhoinv)                ! PREM seismic velocities
     vshomo=SQRT(mu*rhoinv)                        


! Read slab files as perturbation to PREM model.
! Input files are 2D cross sections (r-phi) for each node
! containing perturbation values to be multiplied with PREM velocities

    WRITE(*,*)''
    WRITE(99,*)''

    WRITE(*,*)'reading Tonga slab model ...'
    WRITE(99,*)'reading Tonga slab model ...'

30   FORMAT(1p41e16.7)  ! nz=41 columns 
31   FORMAT(1p39e16.7)  ! nz=39 columns  (without <izfree)
32   FORMAT(1e16.7)

      OPEN(unit=11,file='slabmodel1')

        DO i=0,ny
         DO j=izfree,nz
           READ(11,*)tonga(1,i,j)
           tonga(:,i,j)=tonga(1,i,j) 
         ENDDO
        ENDDO

        tonga(:,:,0:izfree-1)=1.
      CLOSE(11)

    WRITE(*,*)'slab percent.:',MAXVAL(tonga),MINVAL(tonga)     
 
    rhoinv=rhoinv/tonga         ! apply perturbation to PREM for rho
    vphomo=vphomo*tonga         ! apply perturbation to PREM for velocities
    vshomo=vshomo*tonga

    mu=1/rhoinv*vshomo**2       ! obtain 3D elastic parameters including slab
    lam=1/rhoinv*vphomo**2-2*mu

    WRITE(99,*)'   PREM with Tonga slab perturbations: '
    WRITE(99,*)MINVAL(1/rhoinv),' < rho < ',MAXVAL(1/rhoinv)
    WRITE(99,*)MINVAL(SQRT((lam+2*mu)*rhoinv)),' < vp < ', &
               MAXVAL(SQRT((lam+2*mu)*rhoinv))
    WRITE(99,*)MINVAL(SQRT(mu*rhoinv)),' < vs < ',MAXVAL(SQRT(mu*rhoinv))     
    WRITE(99,*)''
    WRITE(99,*)' pert (vs):', &
                 MAXVAL((SQRT((lam(:,:,izfree:nz)+2* &
                 mu(:,:,izfree:nz))*rhoinv(:,:,izfree:nz))- &
                 vshomo(:,:,izfree:nz))/vshomo(:,:,izfree:nz) )       
                           
!=============================================================================


    ENDIF ! different models

! saving background models to file

    IF (savemodel .EQ. 1) THEN  

      WRITE(*,*)' Saving model ...'
      WRITE(99,*)' Saving model ...'

! theta plane

! 2D r-phi section at theta_min
   OPEN(unit=24,file=seisfile(1:lnblnk(seisfile))//'_modelxs',status='unknown')

! 2D r-phi section at theta(isx): through hypocenter
   OPEN(unit=25,file=seisfile(1:lnblnk(seisfile))//'_modelxc',status='unknown')

    DO j=0,ny
     DO k=izfree,nz
      WRITE(24,*)zs(k),1/rhoinv(0,j,k),lam(0,j,k),mu(0,j,k)    
      WRITE(25,*)zs(k),1/rhoinv(isx,j,k),lam(isx,j,k),mu(isx,j,k) 
     ENDDO
    ENDDO
   CLOSE(24)
   CLOSE(25)
 
! 2D r-theta section at phi_min
   OPEN(unit=26,file=seisfile(1:lnblnk(seisfile))//'_modelys',status='unknown')

! 2D r-theta section at phi(isy): through hypocenter
   OPEN(unit=27,file=seisfile(1:lnblnk(seisfile))//'_modelyc',status='unknown')

     DO i=0,nx
      DO k=izfree,nz   
       WRITE(26,*)zs(k),1/rhoinv(i,0,k),lam(i,0,k),mu(i,0,k)
       WRITE(27,*)zs(k),1/rhoinv(i,iplaney,k),lam(i,iplaney,k),mu(i,iplaney,k)
      ENDDO
     ENDDO

   CLOSE(26)
   CLOSE(27)
        
   ENDIF   ! whether to save models or not
        
   WRITE(*,*)' End model'
   WRITE(99,*)' End model'

   RETURN
	
END SUBROUTINE fd3s_model



!########### Additional subroutines ########################################## 

! subroutine PREM to generate radially symmetric earth models

    SUBROUTINE prem	
    IMPLICIT NONE 
    INCLUDE 'common_global.h'
                       
    REAL dist,x
             
    WRITE(*,*)' Generating prem Earth model .... '
    WRITE(99,*)' Generating prem Earth model .... '
    WRITE(99,*)' No. of radii   :  ',nz
    WRITE(99,*)' Max. depth (km): ',(zmax-zmin)/1000

    DO i=0,nz

      dist=zs(i)/1000.   ! Radius (km)
      x=dist/6371.       ! Radius (normalized to x(surface)=1 ) 

	IF(dist.GE.6356)THEN                              ! upper crustal layer
	 ro(i)=2.6
	 vp(i)=5.8
	 vs(i)=3.2
	ELSEIF(dist .LT. 6356 .AND. dist.GE.6346.6)THEN   ! lower crustal layer
	 ro(i)=2.9
	 vp(i)=6.8
	 vs(i)=3.9
	ELSEIF(dist.LT.6346.6.AND.dist.GE.6151)THEN       ! upper mantle
	 ro(i)=2.691+.6924*x
	 vp(i)=4.1875+3.9382*x
	 vs(i)=2.1519+2.3481*x
	ELSEIF(dist.LT.6151.AND.dist.GE.5971)THEN 
	 ro(i)=7.1089-3.8045*x
	 vp(i)=20.3926-12.2569*x
	 vs(i)=8.9496-4.4597*x
	ELSEIF(dist.LT.5971.AND.dist.GE.5771)THEN 
	 ro(i)=11.2494-8.0298*x
	 vp(i)=39.7027-32.6166*x
	 vs(i)=22.3512-18.5856*x
	ELSEIF(dist.LT.5771.AND.dist.GE.5701)THEN 
	 ro(i)=5.3197-1.4836*x
	 vp(i)=19.0957-9.8672*x
	 vs(i)=9.9839-4.9324*x
	ELSEIF(dist.LT.5701.AND.dist.GE.5600)THEN         ! lower mantle  
	 ro(i)=7.9565-6.4761*x+5.5283*x**2-3.0807*x**3
	 vp(i)=29.2766-23.6027*x+5.5242*x**2-2.5514*x**3
	 vs(i)=22.3459-17.2473*x-2.0834*x**2+0.9783*x**3
	ELSEIF(dist.LT.5600.AND.dist.GE.3630)THEN 
	 ro(i)=7.9565-6.4761*x+5.5283*x**2-3.0807*x**3
	 vp(i)=24.9520-40.4673*x+51.4832*x**2-26.6419*x**3
	 vs(i)=11.1671-13.7818*x+17.4575*x**2-9.2777*x**3
	ELSEIF(dist.LT.3630.AND.dist.GE.3480)THEN 
	 ro(i)=7.9565-6.4761*x+5.5283*x**2-3.0807*x**3 
	 vp(i)=15.3891-5.3181*x+5.5242*x**2-2.5514*x**3
	 vs(i)=6.9254+1.4672*x-2.0834*x**2+.9783*x**3
	ELSEIF(dist.LT.3480.AND.dist.GE.1221.5)THEN       ! outer core
	 ro(i)=12.5815-1.2638*x-3.6426*x**2-5.5281*x**3
	 vp(i)=11.0487-4.0362*x+4.8023*x**2-13.5732*x**3
	 vs(i)=0.05
	ELSEIF(dist.LT.1221.5)THEN                        ! inner core  
	 ro(i)=13.0885-8.8381*x**2
	 vp(i)=11.2622-6.3640*x**2
	 vs(i)=3.6678-4.4475*x**2
	ENDIF
        
      ENDDO

      RETURN

END SUBROUTINE prem


!#############################################################################

! Two-layered background model, discontinuity at 240km depth
! below 240km: 10% higher velocities and density

SUBROUTINE homo_2layer

    IMPLICIT NONE
    INCLUDE 'common_global.h'

    REAL dist,x

    WRITE(*,*)' Generating two-layered model .... '
    WRITE(99,*)' Generating two-layered model .... '
    WRITE(99,*)' No. of radii   : ',nz
    WRITE(99,*)' Max. depth (km): ',(zmax-zmin)/1000

    DO i=0,nz

    dist=zs(i)/1000.      

      IF (dist .GT. 6131.)THEN
       ro(i)=rho0
       vp(i)=vp0
       vs(i)=vs0
      ELSEIF(dist.LE.6131.)THEN
       ro(i)=rho0*1.1
       vp(i)=vp0*1.1
       vs(i)=vs0*1.1
      ENDIF

    ENDDO

    RETURN
	
END SUBROUTINE homo_2layer

!#############################################################################
