!*****************************************************************************
!***************** Initialization of FD operator, model space,****************
!******************* boundaries, receiver location, source *******************
!*****************************************************************************

SUBROUTINE fd3s_init

    IMPLICIT NONE
    INCLUDE 'common_global.h'

    REAL x(0:nx),y(0:ny),z(0:nz)                 ! coordinate axes
    REAL mindist                                 ! vertical distance
    REAL xx(0:nx,0:ny,0:nz),yy(0:nx,0:ny,0:nz)   ! cartesian coord.
    REAL zz(0:nx,0:ny,0:nz)                      ! cartesian coord.
    INTEGER lnblnk

! variables for receiver interpolation
    REAL xxloc,yyloc,hyx(1:maxnr),hxx(1:maxnr),hyy(1:maxnr),hxy(1:maxnr)
    REAL hy(1:maxnr),hx(1:maxnr),diffxx(1:maxnr),diffxy(1:maxnr),diffx(1:maxnr)

    WRITE(*,*)' Begin init '
    WRITE(99,*)' Begin init '

!=============================================================================

! derivative operators

    IF(nop .EQ. 2) THEN      ! two-point operator
     g(1)=1.
    ENDIF 

    IF(nop .EQ. 4)THEN       ! four-point operator
     g(1)=.0416666666
     g(2)=-1.125
    ENDIF

    IF(nop .EQ. 6)THEN       ! six-point operator
     g(1)=-0.00468750000000
     g(2)= 0.06510416666667
     g(3)=-1.17187500000000
    ENDIF

    IF(nop .EQ. 8)THEN       ! eight-point operator
     g(1)=0.1344048E-02
     g(2)=-0.1324718E-01
     g(3)=0.8948020E-01
     g(4)=-0.1211563E+01
    ENDIF

    DO i=nop/2+1,nop         !  operator symmetry
     g(i) = -g(nop-i+1)
    ENDDO

!=============================================================================

! source time function  

    IF(aa .NE. 0.)THEN                ! smooth wavelet as source time function
      
      CALL gausswavelet(dt,aa,so)     ! choose between wavelets by 

      smax=0.                         ! normalize wavelet
      DO i=1,nt
        IF(ABS(so(i)).GT.smax)THEN   
          smax=ABS(so(i))
        ENDIF               
      ENDDO
      so=so/smax
                         
    ELSE                              ! delta pulse as source time function
      so=0.
      so(it0:it0+1)=1.        

    ENDIF
        
! save source time function to file
 
      OPEN(unit=30,file=seisfile(1:lnblnk(seisfile))//'_src',status='unknown')
        DO i=1,nt
          WRITE(30,*)so(i)        
        ENDDO
      CLOSE(30)


!=============================================================================

! initialize source moment tensor 

    moment(4)= -SIN(dip)*COS(rake)*SIN(2*strike) &     ! M_theta,theta
     	        -SIN(2*dip)*SIN(rake)*SIN(strike)**2

    moment(5)=  SIN(dip)*COS(rake)*SIN(2*strike) &     ! M_phi,phi
     		-SIN(2*dip)*SIN(rake)*COS(strike)**2

    moment(6)=  SIN(2*dip)*SIN(rake)                   ! M_r,r

    moment(7)=  SIN(dip)*COS(rake)*COS(2*strike) &     ! M_theta,phi
     		+.5*SIN(2*dip)*SIN(rake)*SIN(2*strike)

    moment(8)= -COS(dip)*COS(rake)*COS(strike)  &      ! M_theta,r
     		-COS(2*dip)*SIN(rake)*SIN(strike) 

    moment(9)= -COS(dip)*COS(rake)*SIN(strike)  &      ! M_phi,r
     		+COS(2*dip)*SIN(rake)*COS(strike)

!=============================================================================
        
! Initialization of geometrical variables    

! convert to radians
    cc=pi/180.         
    ccinv=1./cc
    xmin=xmin*cc  ! minimal theta value, i.e. northermost
    xmax=xmax*cc
    ymin=ymin*cc  ! minimal phi value, i.e. westernmost
    ymax=ymax*cc

! define model space range
    xrange=xmax-xmin   ! (rad)
    yrange=ymax-ymin   ! (rad)
    zrange=zmax-zmin   ! (m)

! define grid spacing
    dx=xrange/nx     ! (rad)   
    dy=yrange/ny     ! (rad)   
    dz=zrange/nz     ! (m)  

! inverse grid spacing for faster FD computation
    dxinv=1./dx  
    dyinv=1./dy  
    dzinv=1./dz  

!=============================================================================

! coordinates in theta direction

    OPEN(unit=30,file=seisfile(1:lnblnk(seisfile))//'_xco',status='unknown')
   
      DO ix=0,nx 
        x(ix)=ix*dx  
        xs(ix)=xmin+x(ix)              ! theta coord. axis of model space (rad)
	WRITE(30,*)xs(ix)
      ENDDO

    CLOSE(30)


! coordinates in phi direction

    OPEN(unit=30,file=seisfile(1:lnblnk(seisfile))//'_yco',status='unknown')
     
      DO iy=0,ny
        y(iy)=iy*dy
        ys(iy)=ymin+y(iy)              ! phi coord. axis of model space (rad)
        WRITE(30,*)ys(iy)
      ENDDO

    CLOSE(30)


! coordinates in vertical direction

    DO iz=0,nz 
      z(iz)=iz*dz   ! includes overlap region
                                        ! for MPI exchange 
      zs(iz)=zmax-z(iz)+izfree*dz       ! vert. coord. axis of one node (m)
    ENDDO

! save to file for each node
        
    OPEN(unit=30,file=seisfile(1:lnblnk(seisfile))//'_zco',status='unknown')

      DO iz=izfree,nz
        WRITE(30,*)zs(iz)
      ENDDO
    
    CLOSE(30)

!=============================================================================

! initialization of 3D spherical scaling factors for faster FD computations
 
    DO i=0,nx
     sininv1(i,:,:)=1./SIN(xs(i))                    ! for sigma_(i,i),  
     cottheta1(i,:,:)=COS(xs(i))*sininv1(i,:,:)      !sigma_(phi,r),v_phi,v_r

     sininv2(i,:,:)=1./SIN(xs(i)+dx/2)               ! for sigma_(theta,r)  
     cottheta2(i,:,:)=COS(xs(i)+dx/2)*sininv2(i,:,:) !sigma_(theta,phi),v_theta
    ENDDO

!=============================================================================

! calculate receiver location interpolation values  

       DO k=1,nr                  ! for each receiver

        xxloc=recloc(1,k)/180.*pi ! theta coord. of receivers
        yyloc=recloc(2,k)/180.*pi ! phi coord. of receivers
 
        DO i=0,nx-1               ! find closest theta grid values

!~~~~~~~~ theta velocity component ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ 
                                  ! grid of v_theta in theta direction 
                                  ! is shifted by +dx/2 compared 
                                  ! to normal stresses

         IF(xxloc .GE. xs(i)+dx/2 .AND. &
          xxloc .LT. xs(i+1)+dx/2 ) THEN

          ilowx(k)=i              ! neighboring grid point index to the north
          iupx(k) =i+1            ! neighboring grid point index to the south
          diffxx(k)=xs(i+1)-xs(i) ! distance between adjacent grid points (rad)
          hxx(k)=xxloc-xs(i)-dx/2 ! distance receiver<->northern neighbor (rad)

         ENDIF


!~~~~~~~~ phi velocity component ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
                                  ! grid of v_phi in theta direction 
                                  ! is colocated with normal stresses

         IF(xxloc .GE. xs(i) .AND. &
          xxloc .LT. xs(i+1) ) THEN 

          ilowy(k)=i              ! neighboring grid point index to the north
          iupy(k) =i+1            ! neighboring grid point index to the south
          diffxy(k)=xs(i+1)-xs(i) ! distance between adjacent grid points (rad)
          hxy(k)=xxloc-xs(i)      ! distance receiver<->northern neighbor (rad)

         ENDIF

!~~~~~~~~ vertical velocity component~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
                                 ! grid of v_r in theta direction 
                                 ! is colocated with normal stresses

         IF(xxloc .GE. xs(i) .AND. &
          xxloc .LT. xs(i+1) ) THEN

          ilowz(k)=i              ! neighboring grid point index to the north
          iupz(k) =i+1            ! neighboring grid point index to the south
          diffx(k)=xs(i+1)-xs(i)  ! distance between adjacent grid points (rad)
          hx(k)=xxloc-xs(i)       ! distance receiver<->northern neighbor (rad)

         ENDIF

        ENDDO  ! end search along theta direction

     xqotx(k)=hxx(k)/diffxx(k)    ! v_th: relat. distance to northern neighbor 
     xqotminx(k)=1-xqotx(k)       ! v_th: relat. distance to southern neighbor
     xqoty(k)=hxy(k)/diffxy(k)    ! v_ph: relat. distance to northern neighbor
     xqotminy(k)=1-xqoty(k)       ! v_ph: relat. distance to southern neighbor
     xqotz(k)=hx(k)/diffx(k)      ! v_r: relat. distance to northern neighbor
     xqotminz(k)=1-xqotz(k)       ! v_r: relat. distance to southern neighbor


     DO j=0,ny-1                  ! find closest phi grid values

!~~~~~~~~ theta velocity component ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
                                  ! grid of v_theta in phi direction 
                                  ! is colocated with normal stresses

         IF(yyloc .GE. ys(j) .AND. &
          yyloc .LT. ys(j+1) ) THEN

          jlowx(k)=j              ! neighboring grid point index to the west
          jupx(k) =j+1            ! neighboring grid point index to the east
          diffxx(k)=ys(j+1)-ys(j) ! distance between adjacent grid points (rad)
          hyx(k)=yyloc-ys(j)      ! distance receiver<->western neighbor (rad)

         ENDIF     

!~~~~~~~~ phi velocity component ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ 
                                  ! grid of v_theta in phi direction 
                                  ! is shifted by +dy/2 compared 
                                  ! to normal stresses

         IF(yyloc .GE. ys(j)+dy/2 .AND. &
          yyloc .LT. ys(j+1)+dy/2 ) THEN

          jlowy(k)=j              ! neighboring grid point index to the west
          jupy(k) =j+1            ! neighboring grid point index to the east
          diffxy(k)=ys(j+1)-ys(j) ! distance between adjacent grid points (rad)
          hyy(k)=yyloc-ys(j)-dy/2 ! distance receiver<->western neighbor (rad)
 
         ENDIF

!~~~~~~~~ vertical velocity component ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
                                  ! grid of v_theta in phi direction 
                                  ! is colocated with normal stresses

         IF(yyloc .GE. ys(j) .AND. &
          yyloc .LT. ys(j+1) ) THEN

          jlowz(k)=j              ! neighboring grid point index to the west
          jupz(k) =j+1            ! neighboring grid point index to the east
          diffx(k)=ys(j+1)-ys(j)  ! distance between adjacent grid points (rad)
          hy(k)=yyloc-ys(j)       ! distance receiver<->western neighbor (rad)
 
         ENDIF
  
     ENDDO   ! end search along phi direction
                                 
     yqotx(k)=hyx(k)/diffxy(k)    ! v_th: relat. distance to western neighbor 
     yqotminx(k)=1-yqotx(k)       ! v_th: relat. distance to eastern neighbor 
     yqoty(k)=hyy(k)/diffxy(k)    ! v_ph: relat. distance to western neighbor 
     yqotminy(k)=1-yqoty(k)       ! v_ph: relat. distance to eastern neighbor 
     yqotz(k)=hy(k)/diffx(k)      ! v_r: relat. distance to western neighbor 
     yqotminz(k)=1-yqotz(k)       ! v_r: relat. distance to eastern neighbor 

     ENDDO  ! end loop over different receiver loactions

!=============================================================================
  
! spatial source extent 
        
    zzzs=zmax-zzs   ! source radius

! nonzero source halfwidth: 3D gaussian function 

    IF(alpha .NE. 0.)THEN    

    alpha=1./(alpha*alpha)

    DO ix=0,nx 
      DO iy=0,ny
        DO iz=0,nz
          xx(ix,iy,iz)=zs(iz)*SIN(xs(ix))*COS(ys(iy)) &
                       -zzzs*SIN(xxs*cc)*COS(yys*cc)
          yy(ix,iy,iz)=zs(iz)*SIN(xs(ix))*SIN(ys(iy)) &
                       -zzzs*SIN(xxs*cc)*SIN(yys*cc)
          zz(ix,iy,iz)=zs(iz)*COS(xs(ix))-zzzs*COS(xxs*cc) 

          gauss(ix,iy,iz)=EXP(-alpha*xx(ix,iy,iz)**2.  &
                          -alpha*yy(ix,iy,iz)**2.-alpha*zz(ix,iy,iz)**2.)
        ENDDO
      ENDDO
    ENDDO

! point source

    ELSE    

    gauss=0.
        
    isx=MAXVAL(MINLOC(ABS(xs-xxs/180.*pi)))-1        ! source theta index
    isy=MAXVAL(MINLOC(ABS(ys-yys/180.*pi)))-1        ! source phi index

! check whether source is inside this node

      mindist=zs(0)-zs(nz)
        DO i=0,nz
          IF (ABS(zmax-zs(i)-zzs) .LE. mindist) THEN
            isz=i                                    ! source vertical index
                                                     ! within the node
            mindist=ABS(zmax-zs(i)-zzs) 
          ENDIF
        ENDDO

      zsrc=isz 

      gauss(isx,isy,isz)=1.

      WRITE(*,*)'Source centered in grid point',isx,isy,zsrc
      WRITE(*,*)'Source coordinates :',xs(isx)*180/pi,ys(isy)*180/pi,zs(isz) 
 
      WRITE(99,*)'Source centered in grid point',isx,isy,zsrc
      WRITE(99,*)'Source coordinates :',xs(isx)*180/pi,ys(isy)*180/pi,zs(isz)
      WRITE(99,*)MINVAL(gauss),' < gauss < ',MAXVAL(gauss)


    ENDIF ! end point source initialization

!=============================================================================


! absorbing boundaries

    CALL absorb_model

!=============================================================================
    
    WRITE(*,*)' End init '
    WRITE(99,*)' End init '
    RETURN
	

END SUBROUTINE fd3s_init



!########### Additional subroutines ########################################## 


! Gaussian wavelet

SUBROUTINE gausswavelet(dt3,f0,source)
        
    IMPLICIT NONE

    INCLUDE 'common_global.h'

    REAL f0,dt3,arg,t0
    REAL source(maxnt),ntgs,arggs,tgs
 
    ntgs=2*f0/dt3
    t0=f0/dt3
    arggs=4.4/f0

    DO it=1,ntgs
      tgs=(it-t0)*dt3
      source(it)=EXP(-(arggs*tgs)**2 )
    ENDDO 

    RETURN

END SUBROUTINE gausswavelet    


!#############################################################################


! Ricker wavelet (First time derivative of Gaussian wavelet)

SUBROUTINE ricker(dt3,f0,source)

    IMPLICIT NONE
    INCLUDE 'common_global.h'

    INTEGER ntri
    REAL f0,dt3,argri,t0,tri
    REAL source(maxnt)

    ntri=2*f0/dt3        
    t0=f0/dt3
    argri=4.4/f0

    DO it=1,ntri
      tri=(it-t0)*dt3
      source(it)=-2.*argri*tri*EXP(-(argri*tri)**2.)
    ENDDO
        
    RETURN

END SUBROUTINE ricker              


!#############################################################################

SUBROUTINE absorb_model

    IMPLICIT NONE	
    INCLUDE 'common_global.h'
    
    REAL aabs,tmp(0:nx,0:ny,0:nz)
        
    WRITE(99,*)'absorption model:'

! taper parameters
   
    aabs=3.*nabs
	
! initialise spatial window

    window=1.
    tmp=1.

    DO i=1,nabs
      taper(nabs+1-i)=EXP(-1./aabs**2*(i-1)**2)   ! Gaussian taper function
    ENDDO

! initialize 3D damping function window
    DO i=1,nabs 
      window(i-1,:,:)=taper(i)        ! define damping for theta_min boundary
      window(nx+1-i,:,:)=taper(i)     ! define damping for theta_max boundary
      tmp(:,i-1,:)=taper(i)           ! define damping for phi_min boundary 
      tmp(:,ny+1-i,:)=taper(i)        ! define damping for phi_max boundary 
    ENDDO

    window=window*tmp    ! construct 3D damping function for side boundaries

    tmp=1.

      DO i=1,nabs            
        tmp(:,:,nz+1-i)=taper(i)
      ENDDO      

    window=window*tmp/MAXVAL(window)   ! 3D normalized damping function 
    WRITE(99,*)MINVAL(window),' < window < ',MAXVAL(window)
	
    RETURN

END SUBROUTINE absorb_model

!#############################################################################




















